<#
.SYNOPSIS
   Search and optionally delete files based on specified strings and directories.

.DESCRIPTION
   This PowerShell script searches for files containing specified strings in their names within the specified top-level directories. It can also delete the files that match the criteria.
   
   Originally designed to identify duplicate OneDrive created files (e.g. "*-COMPUTERNAME*") and facilitate cleanup.

.NOTES
   File: Search.ps1
   Author: R. Schmitz
   Created: 2023/09/18
   Updated: 2023/09/18
#>

param (
    [string[]]$stringsToSearch = @("-${env:COMPUTERNAME}"), # Default string to search, comma-delimited
    [string[]]$topLevelDirectories = @("${env:ONEDRIVE}"), # Default top-level directory(ies) to recursively search, comma-delimited
	[string[]]$logFileName = @("Search"), # Default log file prefix
	[int]$maxLogFiles = 5, # Default max number of log files with $logFileName
    [switch]$deleteFiles = $false # Switch parameter with a default value of $false (no delete match), $true will delete matches
)

# Get the computer name
$computerName = $env:COMPUTERNAME

# Define the output folder path based on the computer name and Desktop / OneDrive status
$logFolderPath = Join-Path -Path ([System.Environment]::GetFolderPath('Desktop')) -ChildPath "${computerName}\Search"

# Create the output folder if it doesn't exist
if (-not (Test-Path -Path $logFolderPath -PathType Container)) {
    New-Item -Path $logFolderPath -ItemType Directory
}

# List all log files in the folder
$logFiles = Get-ChildItem -Path $logFolderPath -Filter "$logFileName*.txt" | Sort-Object -Property LastWriteTime

# Check if there are more log files than the maximum allowed
if ($logFiles.Count -gt $maxLogFiles) {
    # Determine how many files need to be deleted
    $filesToDelete = $logFiles.Count - $maxLogFiles

    # Delete the oldest files
    for ($i = 0; $i -lt $filesToDelete; $i++) {
        $logFiles[$i] | Remove-Item -Force
    }
}

# Start a new transcript with a timestamp
$logFilePath = Join-Path -Path $logFolderPath -ChildPath ("${logFileName}_$(Get-Date -Format 'yyyyMMddHHmmss').txt")
Start-Transcript -Path $logFilePath

# Function to search and delete files
function Search-Files {
    param (
        [string]$searchString,
        [string]$directory
    )

    # Search for files matching the search string
    $files = Get-ChildItem -Path $directory -Recurse -ErrorAction SilentlyContinue | Where-Object { $_.Name -like "*$searchString*" }

    foreach ($file in $files) {
        Write-Output "Found file: $($file.FullName)"

        if ($deleteFiles -eq $true) {
            # Delete the file if the deleteFiles flag is set to $true
            Remove-Item -Path $file.FullName -Force
            Write-Output "Deleted file: $($file.FullName)"
        }
    }
}

# Main script
foreach ($searchString in $stringsToSearch) {
    foreach ($directory in $topLevelDirectories) {
        Write-Output "Searching for files containing '$searchString' in directory '$directory'..."
        Search-Files -searchString $searchString -directory $directory
    }
}

# Stop transcription to finish capturing output
Stop-Transcript
