<#
.SYNOPSIS
   Runs scripts when when a specific process is not running and if OneDrive is not syncing.

.DESCRIPTION
   Runs script(s) only when a particular process is not running and OneDrive is not syncing to prevent
   sync issues.
   
.NOTES
   File: Check-Then-Run.ps1
   Author: R. Schmitz
   Created: 2023/09/22
   Updated: 2023/09/22
#>

$processName = "eqgame.exe"
$scriptPath = Join-Path -Path ([System.Environment]::GetFolderPath('Desktop')) -ChildPath "${env:COMPUTERNAME}"
$delayMinutes = 60  # Desired delay in minutes

# File paths for timestamp and lock files
$timestampFile = "${scriptPath}\Check-Then-Run-Timestamp.txt"
$lockFile = "${scriptPath}\Check-Then-Run-LockFile.lock"

# Function to get the last execution timestamp
function Get-LastExecutionTimestamp($tsFile) {
    if (Test-Path $tsFile) {
        return Get-Content $tsFile
    }
    return $null
}

# Function to check if OneDrive is syncing
function IsOneDriveSyncing() {
    $OneDriveSyncStatus = Get-ItemProperty -Path "$env:USERPROFILE\OneDrive\*.log" | Where-Object { $_.Name -like "*sync*.log" }
    if ($OneDriveSyncStatus) {
        return $OneDriveSyncStatus.Length -gt 0
    }
    return $false
}

# Function to set the current timestamp as the last execution timestamp
function Set-LastExecutionTimestamp($tsFile) {
    (Get-Date).ToString() | Set-Content $tsFile
}

# Check for the presence of a lock file
if (Test-Path $lockFile) {
    # Another instance is already running; exit gracefully
    exit
}

# Create a lock file to prevent other instances from running concurrently
New-Item -Path $lockFile -ItemType file

# Loop indefinitely
while ($true) {
    $lastExecutionTimestamp = Get-LastExecutionTimestamp($timestampFile)

    if (!$lastExecutionTimestamp -or ((Get-Date) - [DateTime]$lastExecutionTimestamp).TotalMinutes -ge $delayMinutes) {
        $processes = Get-Process | Where-Object { $_.ProcessName -eq $processName }

        if ($processes.Count -eq 0) {
            # Check if OneDrive is currently syncing
            $isSyncing = IsOneDriveSyncing

            if (-not $isSyncing) {
                # Execute your batch or PowerShell scripts located on the desktop folder
                Invoke-Expression -Command (Join-Path -Path $scriptPath -ChildPath "SyncMulti.bat") # Make sure to update
				#Invoke-Expression -Command (Join-Path -Path $scriptPath -ChildPath "SyncNext.bat")

                # Set the current timestamp as the last execution timestamp
                Set-LastExecutionTimestamp($timestampFile)
												
				Write-Host "Exiting the loop..."
				break  # Exit the while loop after executing the script
            }
        }
    }

    else {
        # Sleep for a shorter duration between checks (e.g., 5 minutes)
        Start-Sleep -Seconds 300  # 300 seconds = 5 minutes
    }
}

# Remove the lock file when the script finishes
Remove-Item -Path $lockFile
